<?php

namespace App\Controladores;

use App\Modelos\Pedidos;
use App\Modelos\PedidosArticulos;
use App\Modelos\PedidosDocumentos;
use Core\Controller;
use Core\Librerias\CarritoVentasEditar;
use Core\Librerias\Http;
use Core\Librerias\Module;
use Core\Helpers\Moneda;
use Core\Vista;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

final class PedidoseditarControlador extends Controller
{
    private $array_errors = []; 
    private $notas_entrega_model;
    private $carritoVentas;
    private $divisa_master;

    public function __construct()
    {
        parent::__construct();
        $this->notas_entrega_model = new Pedidos();
        $this->carritoVentas = new CarritoVentasEditar();
        $this->divisa_master  = $this->notas_entrega_model->obtenerDivisaMaster();
    }

    public function verFactura(int $id_factura)
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET') {

            $id_factura = filter_var($id_factura, FILTER_SANITIZE_NUMBER_INT);
            $factura = $this->notas_entrega_model->obtenerFactura($id_factura);

            $doc = $this->carritoVentas->obtenerDocumentosPendientesPorPagarPorCliente($factura['pe_cliid']);
            $pagot=Moneda::moneda($doc['total'],$this->divisa_master['div_locale'], $this->divisa_master['div_simbolo']);


                $clp=$this->carritoVentas->obtenerCliente($factura['pe_cliid']);
                $rt=str_replace('(','',$clp['cli_telefono']);
                $rt=str_replace(')','',$rt);
                $rt=str_replace('-','',$rt);
                $rt=str_replace(' ','',$rt);

            if ($factura) {
                return Vista::view('pedidoseditar/ver', [
                    'factura'               => $factura,
                    'doc_pendientes'        => $this->carritoVentas->totalDocumentosPendientes($factura['pe_cliid']),
                    'total_doc_pendientes'  => $this->carritoVentas->obtenerDocumentosPendientesPorPagarPorCliente($factura['pe_cliid']),
                    'empleado'              => $this->carritoVentas->obtenerEmpleado($factura['pe_empleadoid']),
                    'pagot'                 => $pagot,
                    'telefonocli'           => $rt,
                    'empresa'               => $this->carritoVentas->obtenerDatosEmpresa(),
                    'cliente'               => $this->carritoVentas->obtenerCliente($factura['pe_cliid']),
                    'tipos_de_documentos'   => $this->carritoVentas->obternerTiposDeDocumentos(),
                    'impuestos'             => $this->carritoVentas->obtenerImpuestos(),
                    'paises'                => $this->carritoVentas->obtenerPaises(),
                    'almacenes'             => $this->carritoVentas->obtenerAlmacenesEmpleado(),
                    'series'                => $this->carritoVentas->obtenerSeriesEmpleado(),
                    'divisas'               => $this->carritoVentas->obtenerDivisasParaFacturar()
                ]);
            } else {
                header('Location: ' . ruta_base() . 'facturacion');
            }
        }
    }

    public function obtenerFactura(int $id_factura)
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET' && isAjax()) {

            $result = $this->carritoVentas->obtenerFactura('app_pedidos', 'peeditar', $id_factura);

            if (isset($result['errors'])) {
                return Http::json_response(['errors' => $result['errors']]);
            } else {
                return Http::json_response($result);
            }
        }
    }

    public function predeterminarDivisa()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_GET['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_GET['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_divisa          = filter_input(INPUT_GET, 'id_divisa', FILTER_VALIDATE_INT);
                $id_factura         = filter_input(INPUT_GET, 'id_factura', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_GET, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_divisa) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->predeterminarDivisa('peeditar', $id_factura, $data_factura['pe_cliid'], $id_divisa, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function agregarProductoBarcode()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_POST['producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_almacen']))
                $this->array_errors[] = 'El Almacen es obligatorio';
            if (empty($_POST['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $producto           = filter_input(INPUT_POST, 'producto', FILTER_SANITIZE_STRIPPED);
                $id_almacen         = filter_input(INPUT_POST, 'id_almacen', FILTER_VALIDATE_INT);
                $id_divisa          = filter_input(INPUT_POST, 'id_divisa', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_almacen && $id_divisa) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->agregarProductoBarcode('app_pedidos', 'peeditar', $id_factura, $data_factura['pe_cliid'], $producto, $id_almacen, $id_divisa, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function agregarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_almacen']))
                $this->array_errors[] = 'El Almacen es obligatorio';
            if (empty($_POST['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $id_almacen         = filter_input(INPUT_POST, 'id_almacen', FILTER_VALIDATE_INT);
                $id_divisa          = filter_input(INPUT_POST, 'id_divisa', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_producto && $id_almacen && $id_divisa) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->agregarProducto('app_pedidos', 'peeditar', $id_factura, $data_factura['pe_cliid'], $id_producto, $id_almacen, $id_divisa, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarPrecio()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $precio             = filter_input(INPUT_POST, 'precio', FILTER_SANITIZE_STRING) ?? 0;
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->actualizarPrecio('peeditar', $id_factura, $data_factura['pe_cliid'], $unique, $precio, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarCantidad()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['cantidad']))
                $this->array_errors[] = 'El Cantidad es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $cantidad           = filter_input(INPUT_POST, 'cantidad', FILTER_VALIDATE_FLOAT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_producto && $cantidad) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->actualizarCantidad('peeditar', 'app_pedidos', $id_factura, $data_factura['pe_cliid'], $id_producto, $cantidad, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarDescripcion()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['descripcion']))
                $this->array_errors[] = 'La Descripción es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRING);
                $descripcion        = $this->solicitud->sanitize($_POST['descripcion'], FILTER_SANITIZE_STRING);

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->actualizarDescripcion('peeditar', $id_factura, $unique, $descripcion);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarNuevoIva()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_iva']))
                $this->array_errors[] = 'El IVA es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $id_iva             = filter_input(INPUT_POST, 'id_iva', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_iva) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->aplicarNuevoIva('peeditar', $id_factura, $data_factura['pe_cliid'], $unique, $id_iva, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarDescuentoProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $descuento          = filter_input(INPUT_POST, 'descuento', FILTER_SANITIZE_STRING) ?? 0;
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->aplicarDescuentoProducto('peeditar', $id_factura, $data_factura['pe_cliid'], $unique, $descuento, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarDescuentoAdicional()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura'])) {
                $this->array_errors[] = 'La Factura es obligatorio';
            } else {

                $id_factura = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $descuento_global = filter_input(INPUT_POST, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->aplicarDescuentoAdicional('peeditar', $id_factura, $data_factura['pe_cliid'], $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function eliminarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRING);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoVentas->eliminarProducto('peeditar', 'app_pedidos', $id_factura, $data_factura['pe_cliid'], $unique, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function guardarDetallesCliente()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['cli_tipodoc']))
                $this->array_errors[] = 'El campo Cedula de identidad es obligatorio';
            if (empty($_POST['cli_dni']))
                $this->array_errors[] = 'El campo Tipo de documento es obligatorio';
            if (empty($_POST['cli_razon_social']))
                $this->array_errors[] = 'El campo Razón social es obligatorio';
            if (empty($_POST['cli_direccion']))
                $this->array_errors[] = 'El campo Dirección es obligatorio';

            if (empty($this->array_errors)) {

                $cli_id             = filter_input(INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT);
                $cli_tipodoc        = filter_input(INPUT_POST, 'cli_tipodoc', FILTER_SANITIZE_NUMBER_INT);
                $cli_dni            = filter_input(INPUT_POST, 'cli_dni', FILTER_SANITIZE_STRING);
                $cli_nombres        = $this->solicitud->sanitize($_POST['cli_razon_social'], FILTER_SANITIZE_STRING);
                $cli_direccion      = $this->solicitud->sanitize($_POST['cli_direccion'], FILTER_SANITIZE_STRING);
                $cli_pais           = filter_input(INPUT_POST, 'cli_paisid', FILTER_SANITIZE_NUMBER_INT);
                $cli_estado         = $this->solicitud->sanitize($_POST['cli_estado'], FILTER_SANITIZE_STRING);
                $cli_ciudad         = $this->solicitud->sanitize($_POST['cli_ciudad'], FILTER_SANITIZE_STRING);
                $cli_codigopostal   = filter_input(INPUT_POST, 'cli_codigopostal', FILTER_SANITIZE_STRING);

                $result = $this->carritoVentas->editarDetallesCliente($cli_id, [
                    'cli_tipodocid'     => $cli_tipodoc,
                    'cli_dni'           => $cli_dni,
                    'cli_razon_social'  => $cli_nombres,
                    'cli_direccion'     => $cli_direccion,
                    'cli_paisid'        => $cli_pais,
                    'cli_estado'        => $cli_estado,
                    'cli_ciudad'        => $cli_ciudad,
                    'cli_codpostal'     => $cli_codigopostal
                ]);

                if (isset($result['errors'])) {
                    return Http::json_response(['errors' => $result['errors']]);
                } else {
                    return Http::json_response($result);
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function enviarDocumento()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'Acción prohibida';
            if (empty($_POST['descripcion']))
                $this->array_errors[] = 'El campo Descripcion es obligatorio.';
            if (empty($_FILES['documento']))
                $this->array_errors[] = 'El documento es obligatorio.';

            if (empty($this->array_errors)) {

                $id_factura = filter_var($_POST['id_factura'], FILTER_VALIDATE_INT);
                $descripcion = $this->solicitud->sanitize($_POST['descripcion'], FILTER_SANITIZE_STRING);

                if ($id_factura) {
                    $result = $this->carritoVentas->enviarDocumento('app_pedidos', $id_factura, $_FILES, $descripcion);
                    if (isset($result['errors'])) {
                        return Http::json_response(['errors' => $result['errors']]);
                    } else {
                        return Http::json_response($result);
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    //METODOS PROPIO DEL CONTRALADOR

    public function ventaEnEsperaEditar()
    {
        if ($_SERVER['REQUEST_METHOD'] == 'POST' && isAjax()) {

            $id_serie       = filter_input(INPUT_POST, 'id_serie', FILTER_SANITIZE_NUMBER_INT);
            $id_factura     = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $fecha          = filter_input(INPUT_POST, 'fecha', FILTER_SANITIZE_STRING);
            $observaciones  = $this->solicitud->sanitize($_POST['observaciones']);

            if (!empty($id_factura)) {

                if (empty($id_serie))
                    $this->array_errors[] = 'El campo Serie es obligatorio.';
                if (empty($fecha))
                    $this->array_errors[] = 'El campo fecha es obligatorio.';

                if (empty($this->array_errors)) {

                    if ($this->esEditable($id_factura)) {

                        $factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                        $serie = $this->carritoVentas->obtenerSerie($id_serie);
                        $cliente = $this->carritoVentas->obtenerCliente($factura['pe_cliid']);

                        if ($factura['pe_estatus'] == 1) {
                            $this->array_errors[] = 'El pedido a sido pagada en su totalidad por lo tanto no puede ser modificada';
                        } else {

                            $notas_entrega_articulos_model = new  PedidosArticulos();
                            $contenido_carrito = $this->carritoVentas->contenidoCarrito('peeditar', $id_factura);
                            $date = date('Y-m-d');
                            $fecha = date('Y-m-d', strtotime($date)) . ' ' . date('H:i:s');
                            $total_costo_servicio = 0;
                            $total_procosto = 0;

                            foreach ($contenido_carrito as $row) {
                                if ($row['precio'] == 0 || $row['total'] == 0) {
                                    $this->array_errors[] = "Lo siento el total del producto <strong>{$row['nombre']}</strong> no puede quedar en cero";
                                }
                            }

                            if (empty($this->array_errors)) {

                                if ($this->notas_entrega_model->editar($id_factura, [
                                    'pe_subtotal'           => $_SESSION["peeditar$id_factura"]['subtotal'],
                                    'pe_descuento'          => $_SESSION["peeditar$id_factura"]['descuento'],
                                    'pe_neto'               => $_SESSION["peeditar$id_factura"]['neto'],
                                    'pe_iva'                => $_SESSION["peeditar$id_factura"]['iva'],
                                    'pe_total'              => $_SESSION["peeditar$id_factura"]['total'],
                                    'pe_articulos_total'    => $_SESSION["peeditar$id_factura"]['articulos_cantidad'],
                                    'pe_divid'              => $_SESSION["peeditar$id_factura"]['divisa'],
                                    'pe_serid'              => $serie['ser_id'],
                                    'pe_serie'              => $serie['ser_descripcion'],
                                    'pe_empresaid'          => 1,
                                    //'pe_empleadoid'         => $_SESSION['user_data']['einfo_id'],
                                    'pe_fecha'              => $fecha,
                                    'pe_cliid'              => $cliente['cli_id'],
                                    'pe_env_nombres'        => $cliente['cli_razon_social'],
                                    'pe_env_direccion'      => $cliente['cli_direccion'],
                                    'pe_env_pais'           => $cliente['cli_paisid'],
                                    'pe_env_estado'         => $cliente['cli_estado'],
                                    'pe_env_ciudad'         => $cliente['cli_ciudad'],
                                    'pe_env_codigo_postal'  => $cliente['cli_codpostal'],
                                    'pe_estatus'            => 2,
                                    'pe_observaciones'      => $observaciones
                                ])) {

                                    $notas_entrega_articulos_model->eliminarArticulos($id_factura);

                                    foreach ($contenido_carrito as $row) {

                                        $producto = $this->carritoVentas->obtenerProducto($row['id']);

                                        if ($notas_entrega_articulos_model->guardar([
                                            'peart_peid'            => $id_factura,
                                            'peart_proid'           => $row['id'],
                                            'peart_procodigo'       => $row['codigo'],
                                            'peart_propnombre'      => $row['nombre'],
                                            'peart_procosto'        => $row['costo'],
                                            'peart_costo_servicio'  => $row['costo_servicio'],
                                            'peart_proprecio'       => $row['precio'],
                                            'peart_procantidad'     => $row['cantidad'],
                                            'peart_prodescuento'    => $row['descuento'],
                                            'peart_proneto'         => $row['neto'],
                                            'peart_idiva'           => $row['id_iva'],
                                            'peart_almid'           => $row['id_almacen'],
                                            'peart_proiva'          => $row['iva'],
                                            'peart_ivatotal'        => $row['iva_total'],
                                            'peart_prototal'        => $row['total']
                                        ], true)) {
                                            if ($producto['pro_tipo'] == 3) {
                                                $total_costo_servicio  += $row['costo_servicio'] * $row['cantidad'];
                                            } else {
                                                $total_procosto += $row['costo'] * $row['cantidad'];
                                            }
                                        }
                                    }

                                    $this->notas_entrega_model->editar($id_factura, [
                                        'total_procosto' => $total_procosto,
                                        'total_costo_servicio' => $total_costo_servicio,
                                    ]);

                                    Http::json_response(ruta_base() . 'pedidoseditar/verFactura/' . $id_factura);
                                } else {
                                    $this->array_errors[] = 'Los siento ocurrio un error al guardar la venta';
                                }
                            }
                        }
                    } else {
                        $this->array_errors[] = 'El pedido a sido pagado en su totalidad por lo tanto no puede ser modificado.';
                    }
                }

                Http::json_response(['errors' => $this->array_errors]);
            }
        }
    }

    public function guardarEnvioCliente()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_factura             = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $env_nombre_opcional    = filter_input(INPUT_POST, 'env_nombre_opcional', FILTER_SANITIZE_STRING);
            $env_agencia_transporte = filter_input(INPUT_POST, 'env_agencia_transporte', FILTER_SANITIZE_STRING);
            $env_codigo_seguimiento = filter_input(INPUT_POST, 'env_codigo_seguimiento', FILTER_SANITIZE_STRING);
            $env_direccion          = filter_input(INPUT_POST, 'env_direccion', FILTER_SANITIZE_STRING);
            $env_pais               = filter_input(INPUT_POST, 'env_pais', FILTER_SANITIZE_NUMBER_INT);
            $env_estado             = filter_input(INPUT_POST, 'env_estado', FILTER_SANITIZE_STRING);
            $env_ciudad             = filter_input(INPUT_POST, 'env_ciudad', FILTER_SANITIZE_STRING);
            $env_codigopostal       = filter_input(INPUT_POST, 'env_codigopostal', FILTER_SANITIZE_STRING);

            if (!empty($id_factura)) {

                if (empty($env_agencia_transporte))
                    $this->array_errors[] = 'El campo Agencia de tranporte es obligatorio.';
                if (empty($env_codigo_seguimiento))
                    $this->array_errors[] = 'El campo Codigo de seguimiento es obligatorio.';
                if (empty($env_direccion))
                    $this->array_errors[] = 'El campo Dirección obligatorio.';
                if (empty($env_pais))
                    $this->array_errors[] = 'El campo País es obligatorio.';
                if (empty($env_estado))
                    $this->array_errors[] = 'El campo Estado es obligatorio.';
                if (empty($env_ciudad))
                    $this->array_errors[] = 'El campo Ciudad es obligatorio.';
                if (empty($env_codigopostal))
                    $this->array_errors[] = 'El campo Codigo postal es obligatorio.';

                if (empty($this->array_errors)) {

                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);

                    if ($data_factura['pe_estatus'] != 1) {
                        Http::json_response($this->notas_entrega_model->editar($id_factura, [
                            'pe_env_nombres'            => $env_nombre_opcional ?? $data_factura['pe_env_nombres'],
                            'pe_env_direccion'          => $env_direccion,
                            'pe_env_pais'               => $env_pais,
                            'pe_env_estado'             => $env_estado,
                            'pe_env_ciudad'             => $env_ciudad,
                            'pe_env_codigo_postal'      => $env_codigopostal,
                            'pe_env_agencia_transporte' => $env_agencia_transporte,
                            'pe_env_codigo_seguimiento' => $env_codigo_seguimiento,
                            'pe_env_asegurado'          => 1,
                        ]));
                    } else {
                        $this->array_errors[] = 'La factura no se puede editar ya que ha sigo pagada en su totalidad.';
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    private function esEditable($id_factura)
    {
        $factura = $this->notas_entrega_model->obtenerFactura($id_factura);
        return ($factura['pe_estatus'] == 1) ? false : true;
    }

    public function eniviarFactura()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id     = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $de     = filter_input(INPUT_POST, 'de', FILTER_SANITIZE_EMAIL);
            $para   = filter_input(INPUT_POST, 'para', FILTER_SANITIZE_EMAIL);
            $copia  = filter_input(INPUT_POST, 'copia', FILTER_SANITIZE_EMAIL);
            $firma  = filter_input(INPUT_POST, 'firma', FILTER_SANITIZE_EMAIL);

            if (empty($de))
                $this->array_errors[] = 'El campo De es obligatorio.';
            if (empty($para))
                $this->array_errors[] = 'El campo Para es obligatorio.';

            if (empty($this->array_errors) && !empty($id)) {
                $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);
                $articulos_model    = new PedidosArticulos();
                $articulos_data     = $articulos_model->articulos($factura_data['pe_id']);
                $cliente_data       = $this->carritoVentas->obtenerCliente($factura_data['pe_cliid']);
                $empresa_data       = $this->carritoVentas->obtenerDatosEmpresa();
                $documento_data     = $this->carritoVentas->obtenerDocumento(4);
                // nota de entrega
                $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['pe_correlativo']);
                $mpdf = new \Mpdf\Mpdf([
                    'mode' => 'utf-8',
                    'margin_left'   => 15,
                    'margin_right'  => 15,
                    'margin_top'    => 35,
                    'margin_bottom' => 30,
                    'margin_header' => 15,
                    'margin_footer' => 30,
                    'format' => 'A4',
                    'orientation' => 'P',
                    'default_font_size' => 8,
                    'default_font' => 'helvetica'
                ]);
                $mpdf->SetTitle($mascara);
                $mpdf->SetAuthor($empresa_data['emp_nombre']);
                $mpdf->SetCreator($empresa_data['emp_nombre']);
                $mpdf->SetSubject($mascara);
                $html_header = Vista::viewPDF('pdffacturas/factura/header', [
                    'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
                    'emp_rif'       => $empresa_data['emp_rif'] ?? '',
                    'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                    'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                    'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
                    'emp_estado'    => $empresa_data['emp_estado'] ?? '',
                    'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
                ]);
                $mpdf->SetHTMLHeader($html_header);
                $html_body = Vista::viewPDF('pdffacturas/pedidos/body', [
                    'factura'       => $mascara,
                    'cliente'       => $cliente_data['cli_razon_social'],
                    'direccion'     => $cliente_data['cli_direccion'],
                    'fecha'         => date('Y-m-d', strtotime($factura_data['pe_fecha'])),
                    'cedularif'     => $cliente_data['tipodoc_codigo'] . '' . $cliente_data['cli_dni'],
                    'telefono'      => $cliente_data['cli_telefono'],
                    'div_locale'    => $factura_data['div_locale'],
                    'div_simbolo'  => $factura_data['div_simbolo'],
                    'articulos'     => $articulos_data,
                    'observaciones' => $factura_data['pe_observaciones']
                ]);
                $mpdf->WriteHTML($html_body);
                $html_footer = Vista::viewPDF('pdffacturas/pedidos/footer', [
                    'div_locale'    => $factura_data['div_locale'],
                    'div_simbolo'  => $factura_data['div_simbolo'],
                    'total'         => $factura_data['pe_total'],
                    'iva'           => $factura_data['pe_iva'],
                    'neto'          => $factura_data['pe_neto'],
                    'descuento'     => $factura_data['pe_descuento']
                ]);
                $mpdf->SetHTMLFooter($html_footer);
                $content = $mpdf->Output('', 'S');
                $mail = new PHPMailer(true);
                try {

                    $mail->isSMTP();
                    $mail->Host       = "{$empresa_data['emp_host_correo']}";
                    $mail->SMTPAuth   = true;
                    $mail->Username   = "{$empresa_data['emp_email']}";
                    $mail->Password   = "{$empresa_data['emp_email_password']}";
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
                    $mail->Port       = $empresa_data['emp_port_correo'];

                    $mail->setFrom($de);
                    $mail->addAddress($para);
                    $mail->addReplyTo($de, 'Empresa');

                    if (!empty($copia)) {
                        $mail->addAddress($copia);
                    }

                    $mail->addStringAttachment($content, "$mascara.pdf");

                    $mail->isHTML(true);
                    $mail->Subject = 'PEDIDO';
                    $mail->Body    = "$mascara";

                    $mail->send();

                    Http::json_response("Envio de factura exitosa");
                } catch (Exception $e) {
                    $this->array_errors[] = "Message could not be sent. Mailer Error: {$mail->ErrorInfo}.";
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function pdf($id)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && !empty($id)) {
            $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);
            $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);
            if (!$factura_data) {
                Http::notFound();
            }
            $articulos_model    = new PedidosArticulos();
            $articulos_data     = $articulos_model->articulos($factura_data['pe_id']);
            $cliente_data       = $this->carritoVentas->obtenerCliente($factura_data['pe_cliid']);
            $empresa_data       = $this->carritoVentas->obtenerDatosEmpresa();
            $documento_data     = $this->carritoVentas->obtenerDocumento(4);
            $sucursal = $this->notas_entrega_model->obtenerSucursal($factura_data['pe_surcid']);
            // nota de entrega
            $this->module = new Module();
            $factCompra = $empresa_data['emp_iva'];
            $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['pe_correlativo']);
            $mpdf = new \Mpdf\Mpdf([
                'mode' => 'utf-8',
                'margin_left'   => 15,
                'margin_right'  => 15,
                'margin_top'    => 35,
                'margin_bottom' => 30,
                'margin_header' => 15,
                'margin_footer' => 30,
                'format' => 'A4',
                'orientation' => 'P',
                'default_font_size' => 8,
                'default_font' => 'helvetica'
            ]);
            $mpdf->SetTitle($mascara);
            $mpdf->SetAuthor($empresa_data['emp_nombre']);
            $mpdf->SetCreator($empresa_data['emp_nombre']);
            $mpdf->SetSubject($mascara);
            $html_header = Vista::viewPDF('pdffacturas/factura/header', [
                'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
                'emp_rif'       => $empresa_data['emp_rif'] ?? '',
                'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
                'emp_estado'    => $empresa_data['emp_estado'] ?? '',
                'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
            ]);
            $mpdf->SetHTMLHeader($html_header);
            $html_body = Vista::viewPDF('pdffacturas/pedidos/body', [
                'factura'       => $mascara,
                'sucursal'      => $sucursal['su_nombre'] . '-' . $sucursal['su_direccion'],
                'iva_integrado' => $factCompra,
                'cliente'       => $cliente_data['cli_razon_social'],
                'direccion'     => $cliente_data['cli_direccion'],
                'fecha'         => date('Y-m-d', strtotime($factura_data['pe_fecha'])),
                'cedularif'     => $cliente_data['tipodoc_codigo'] . '' . $cliente_data['cli_dni'],
                'telefono'      => $cliente_data['cli_telefono'],
                'div_locale'    => $factura_data['div_locale'],
                'div_simbolo'  => $factura_data['div_simbolo'],
                'articulos'     => $articulos_data,
                'observaciones' => $factura_data['pe_observaciones']
            ]);
            $mpdf->WriteHTML($html_body);
            if ($factCompra) {
                $subtotal = 0;
                $neto = 0;
                $total = 0;
                foreach ($articulos_data as $row) {

                    $iv = convertir_a_float($row['peart_proprecio']+($row['peart_proprecio'] * ($row['imp_valor'] / 100)))??0;
                    $descuentoprod = convertir_a_float(($row['peart_procantidad'] * $iv) * ($row['peart_prodescuento'] / 100)) ?? 0;
                    $subtotal += convertir_a_float(($row['peart_procantidad'] * $iv)-$descuentoprod);
                    $neto += convertir_a_float(($row['peart_procantidad'] * $iv)-$descuentoprod);
                    $total +=convertir_a_float(($row['peart_procantidad'] * $iv)-$descuentoprod);
                }
                $descuento = convertir_a_float(($total * ($factura_data['pe_descuento'] / 100)))??0;
                $html_footer = Vista::viewPDF('pdffacturas/pedidos/footer', [
                    'div_locale'    => $factura_data['div_locale'],
                    'div_simbolo'  => $factura_data['div_simbolo'],
                    'iva_integrado' => $factCompra,
                    'total'         => $total-$descuento,
                    'iva'           => $factura_data['pe_iva'],
                    'neto'          => $neto,
                    'descuento'     => $factura_data['pe_descuento']
                ]);
            }else{
                $html_footer = Vista::viewPDF('pdffacturas/pedidos/footer', [
                    'div_locale'    => $factura_data['div_locale'],
                    'iva_integrado' => $factCompra,
                    'div_simbolo'  => $factura_data['div_simbolo'],
                    'total'         => $factura_data['pe_total'],
                    'iva'           => $factura_data['pe_iva'],
                    'neto'          => $factura_data['pe_neto'],
                    'descuento'     => $factura_data['pe_descuento']
                ]);
            }
           
            $mpdf->SetHTMLFooter($html_footer);
            $mpdf->Output("$mascara.pdf", \Mpdf\Output\Destination::INLINE);
            exit;
        }
    }

    public function obtnerDocumentos($id_factura)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $facturadocumentos_model = new PedidosDocumentos();
            $id_factura = filter_var($id_factura, FILTER_SANITIZE_NUMBER_INT);
            if (!empty($id_factura)) {
                Http::json_response($facturadocumentos_model->obtenerDocumentos($id_factura));
            }
        }
    }
}
