<?php

namespace App\Controladores;

use App\Modelos\Grupos;
use Core\Controller;
use Core\Librerias\Http;
use Core\Vista;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;

final class GruposControlador extends Controller
{
    private $grupos;
    private $array_errors = [];

    public function __construct()
    {
        parent::__construct();
        $this->grupos = new Grupos();
    }

    public function index()
    {
        return Vista::view('grupos/index');
    }

    public function guardar()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $grup_id            = filter_input(INPUT_POST, 'grup_id', FILTER_SANITIZE_NUMBER_INT);
            $grup_codigo        = filter_input(INPUT_POST, 'grup_codigo', FILTER_SANITIZE_STRING);
            $grup_nombre        = filter_input(INPUT_POST, 'grup_nombre', FILTER_SANITIZE_STRING);
            $grup_descuento     = filter_input(INPUT_POST, 'grup_descuento', FILTER_SANITIZE_NUMBER_INT) ?? 0;
            $grup_descripcion   = filter_input(INPUT_POST, 'grup_descripcion', FILTER_SANITIZE_STRING);

            if (empty($grup_codigo)) {
                $this->array_errors[] = "El campo Codigo es obligatorio.";
            } else if (!is_numeric($grup_codigo)) {
                $this->array_errors[] = "El campo Codigo debe ser de tipo numerico.";
            }

            if (empty($grup_nombre))
                $this->array_errors[] = "El campo Nombre es obligatorio.";
            if (empty($grup_descripcion))
                $this->array_errors[] = "El campo Descripción es obligatorio.";

            if (empty($this->array_errors)) {

                if (empty($grup_id)) {
                    if ($this->grupos->existe('grup_codigo', $grup_codigo)) {
                        $this->array_errors[] = "El codigo $grup_codigo ya se encuentra en uso";
                    }
                } else {
                    $this->grupos->grupoDiferente("grup_codigo='$grup_codigo' and grup_id=$grup_id");
                }

                if (empty($this->array_errors)) {

                    $data_form = [
                        'grup_codigo'      => $grup_codigo,
                        'grup_nombre'      => $grup_nombre,
                        'grup_descuento'   => $grup_descuento,
                        'grup_descripcion' => $grup_descripcion
                    ];

                    if (empty($grup_id)) {
                        $resultado = $this->grupos->guardar($data_form, true);
                    } else {
                        $resultado = $this->grupos->editar($grup_id, $data_form);
                    }

                    Http::json_response($resultado);
                }
            }

            Http::json_response([
                'errors' => $this->array_errors
            ]);
        }
    }

    public function obtenerGrupos()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            Http::json_response($this->grupos->todos());
        }
    }

    public function modificarEstatus()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD']  && isAjax()) {

            $id = filter_input(INPUT_GET, 'id', FILTER_SANITIZE_STRING);

            if ($this->grupos->existe('grup_id', $id)) {
                $estatus = $this->grupos->estatus($id);
                if ($estatus->grup_estatus) {
                    $resultado = $this->grupos->modificarEstatus($id, 0);
                } else {
                    $resultado = $this->grupos->modificarEstatus($id, 1);
                }
                Http::json_response($resultado);
            } else {
                Http::json_response(false);
            }
        }
    }

    public function eliminar()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id = filter_input(INPUT_GET, 'id', FILTER_SANITIZE_STRING);

            if ($this->grupos->existe('grup_id', $id)) {
                if ($this->grupos->chequearForaneo($id)) {
                    $this->array_errors[] = "El grupo no puede ser eliminado ya que se encuentra asociado a un cliente.";
                } else {
                    Http::json_response($this->grupos->eliminar($id));
                }
            } else {
                Http::json_response(false);
            }

            Http::json_response([
                'errors' => $this->array_errors
            ]);
        }
    }

    public function pdf_clientes(int $id)
    {
        $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

        $list_clients = $this->grupos->listaCliente($id);

        if (!$list_clients) {
            Http::notFound();
        }

        $empresa_data       = $this->grupos->obtenerDatosEmpresa();
        $grup       = $this->grupos->estatus($id);
        $array_productos    = [];

        $mpdf = new \Mpdf\Mpdf([
            'mode'              => 'utf-8',
            'margin_left'       => 15,
            'margin_right'      => 15,
            'margin_top'        => 35,
            'margin_bottom'     => 20,
            'margin_header'     => 15,
            'margin_footer'     => 15,
            'format'            => 'A4',
            'orientation'       => 'P',
            'default_font_size' => 8,
            'default_font'      => 'helvetica'
        ]);

        $mpdf->SetTitle('Productos');
        $mpdf->SetAuthor($empresa_data['emp_nombre']);
        $mpdf->SetCreator($empresa_data['emp_nombre']);
        $mpdf->SetSubject('Productos tarifas');

        $html_header = Vista::viewPDF('grupos/pdf/header.php', [
            'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
            'emp_rif'       => $empresa_data['emp_rif'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
            'emp_estado'    => $empresa_data['emp_estado'] ?? '',
            'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
        ]);

        $mpdf->SetHTMLHeader($html_header);

        $html_footer = Vista::viewPDF('grupos/pdf/footer.php', [
            'grupo'    => $grup['grup_nombre'],
        ]);

        $mpdf->SetHTMLFooter($html_footer);

        $html_body = Vista::viewPDF('grupos/pdf/body.php', [
            'clientes'     => $list_clients,
            'grupo'    => $grup['grup_nombre']
        ]);

        $mpdf->WriteHTML($html_body);

        $mpdf->Output("grup_clientes.pdf", \Mpdf\Output\Destination::INLINE);
        exit;
    }

    public function listadoclientesExcel(int $id)
    {
        $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

        $list_clients = $this->grupos->listaCliente($id);

        if (!$list_clients) {
            Http::notFound();
        }

        $empresa_data       = $this->grupos->obtenerDatosEmpresa();
        $grup       = $this->grupos->estatus($id);

        $spreadsheet = new Spreadsheet();

        $spreadsheet->getProperties()->setCreator($empresa_data['emp_nombre'])->setLastModifiedBy('Firma Digital')->setTitle('Listado de Clientes del Grupo '.$grup['grup_nombre'])->setSubject('Office 2007 XLSX Test Document')->setDescription('Test document for Office 2007 XLSX, generated using PHP classes.')->setKeywords('producto tarifa office 2007 openxml php')->setCategory('Listado de Clientes del Grupo '.$grup['grup_nombre']);

        $spreadsheet->getActiveSheet(0)->setCellValue("A1", 'Cliente')->setCellValue('B1', 'Rif')->setCellValue('C1', 'Telefono')->setCellValue('D1', 'Dirección');

        $i = 2;

        foreach ($list_clients as $pro) {

          //  $spreadsheet->getActiveSheet()->setCellValue("A$i", $row);  
                $spreadsheet->getActiveSheet()->setCellValue("A$i", $pro['cli_razon_social'])->setCellValue("B$i", $pro['cli_dni'])->setCellValue("C$i", $pro['cli_telefono'])->setCellValue("D$i", $pro['cli_direccion']);
          

            $i++;
        }


        $spreadsheet->getActiveSheet()->getColumnDimension('A')->setAutoSize(true);
        $spreadsheet->getActiveSheet()->getColumnDimension('B')->setAutoSize(true);
        $spreadsheet->getActiveSheet()->getColumnDimension('C')->setAutoSize(true);
        $spreadsheet->getActiveSheet()->getColumnDimension('D')->setAutoSize(true);

        $spreadsheet->getActiveSheet()->setTitle('Listado de Clientes del Grupo');
        //$spreadsheet->getActiveSheet()->getProtection()->setSheet(true);

        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $spreadsheet->setActiveSheetIndex(0);

        // Redirect output to a client’s web browser (Xlsx)
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="Listado del grupo '.$grup['grup_nombre'].'.xlsx"');
        header('Cache-Control: max-age=0');
        // If you're serving to IE 9, then the following may be needed
        header('Cache-Control: max-age=1');

        // If you're serving to IE over SSL, then the following may be needed
        header('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT'); // always modified
        header('Cache-Control: cache, must-revalidate'); // HTTP/1.1
        header('Pragma: public'); // HTTP/1.0

        $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
        $writer->save('php://output');
        exit;
    }
}
