<?php

namespace App\Controladores;

use App\Modelos\CmpPedidos;
use App\Modelos\CmpPedidosArticulos;
use App\Modelos\CmpPedidosDocumentos;
use App\Modelos\Comprascredito;
use App\Modelos\Comprascreditopagos;
use App\Modelos\MovimientoModel;
use App\Modelos\Productos;
use App\Modelos\Stock;
use Core\Controller;
use Core\Helpers\Moneda;
use Core\Helpers\Notifica;
use Core\Librerias\CarritoComprasEditar;
use Core\Librerias\Http;
use Core\Librerias\Module;
use Core\Vista;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

final class CmppedidoseditarControlador extends Controller
{
    private $array_errors = [];
    private $pedidos_model;
    private $divisa_master;
    private $module;
    private $carritoCompras;

    public function __construct()
    {
        parent::__construct();
        $this->module = new Module();
        $this->notas_entrega_model = new CmpPedidos();
        $this->carritoCompras = new CarritoComprasEditar();
        $this->divisa_master  = $this->notas_entrega_model->obtenerDivisaMaster();
    }

    public function verFactura(int $id_factura)
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET') {

            $id_factura = filter_var($id_factura, FILTER_SANITIZE_NUMBER_INT);
            $factura = $this->notas_entrega_model->obtenerFactura($id_factura);

            if ($factura) {
                $data_factura = [
                    'factura'               => $factura,
                    'empleado'              => $this->carritoCompras->obtenerEmpleado($factura['cmppe_empleadoid']),
                    'ruta'                  => $this->notas_entrega_model->rutaDocumento($factura['cmppe_id']),
                    'anular_doc'            => $this->module->has_module_action_permission('facturacion', 'anular_doc', $_SESSION['user_data']['emp_id']),
                    'empresa'               => $this->carritoCompras->obtenerDatosEmpresa(),
                    'proveedor'             => $this->notas_entrega_model->obtenerProveedor($factura['cmppe_cliid']),
                    'tipos_de_documentos'   => $this->carritoCompras->obternerTiposDeDocumentos(),
                    'impuestos'             => $this->carritoCompras->obtenerImpuestos(),
                    'paises'                => $this->carritoCompras->obtenerPaises(),
                    'almacenes'             => $this->carritoCompras->obtenerAlmacenesEmpleado(),
                    'series'                => $this->carritoCompras->obtenerSeriesEmpleado(),
                    'divisas'               => $this->carritoCompras->obtenerDivisasParaFacturar()
                ];

                if ($factura['cmppe_estatus'] < 3) {
                    return Vista::view('cmppedidoseditar/ver', $data_factura);
                } elseif ($factura['cmppe_estatus'] == 3) {
                    return Vista::view('cmppedidoseditar/anulado', $data_factura);
                }
            } else {
                header('Location: ' . ruta_base() . 'compras');
            }
        }
    }

    public function obtenerFactura(int $id_factura)
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET' && isAjax()) {

            $result = $this->carritoCompras->obtenerFactura('app_cmppedidos', 'cmppeeditar', $id_factura);

            if (isset($result['errors'])) {
                return Http::json_response(['errors' => $result['errors']]);
            } else {
                return Http::json_response($result);
            }
        }
    }

    public function obtenerArticuloProducto(int $id_factura)
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET' && isAjax()) {

            $articuls = new CmpPedidosArticulos();
            $result = $articuls->articulos($id_factura);

            $inventario = 0;
            $cantidad = 0;

            foreach ($result as $row) {
                $inventario += 0;
                $cantidad += $row['cmppeart_procantidad'];
            }
            return Http::json_response([
                'inventario' => $inventario,
                'cantidad' => $cantidad
            ]);
        }
    }

    public function predeterminarDivisa()
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_GET['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_GET['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_divisa          = filter_input(INPUT_GET, 'id_divisa', FILTER_VALIDATE_INT);
                $id_factura         = filter_input(INPUT_GET, 'id_factura', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_GET, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_divisa) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->predeterminarDivisa('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $id_divisa, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function agregarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_almacen']))
                $this->array_errors[] = 'El Almacen es obligatorio';
            if (empty($_POST['id_divisa']))
                $this->array_errors[] = 'La Divisa es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $id_almacen         = filter_input(INPUT_POST, 'id_almacen', FILTER_VALIDATE_INT);
                $id_divisa          = filter_input(INPUT_POST, 'id_divisa', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_producto && $id_almacen && $id_divisa) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->agregarProducto('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $id_producto, $id_almacen, $id_divisa, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarPrecio()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $precio             = filter_input(INPUT_POST, 'precio', FILTER_SANITIZE_STRING) ?? 0;
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->actualizarPrecio('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $unique, $precio, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarCantidad()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['cantidad']))
                $this->array_errors[] = 'El Cantidad es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $id_producto        = filter_input(INPUT_POST, 'id_producto', FILTER_VALIDATE_INT);
                $cantidad           = filter_input(INPUT_POST, 'cantidad', FILTER_VALIDATE_FLOAT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_producto && $cantidad) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->actualizarCantidad('cmppeeditar', 'app_cmppedidos', $id_factura, $data_factura['cmppe_cliid'], $id_producto, $cantidad, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarDescripcion()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'Accion prohibida';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['descripcion']))
                $this->array_errors[] = 'La Descripción es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRING);
                $descripcion        = $this->solicitud->sanitize($_POST['descripcion'], FILTER_SANITIZE_STRING);

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->actualizarDescripcion('cmppeeditar', $id_factura, $unique, $descripcion);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarNuevoIva()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';
            if (empty($_POST['id_iva']))
                $this->array_errors[] = 'El IVA es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $id_iva             = filter_input(INPUT_POST, 'id_iva', FILTER_VALIDATE_INT);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura && $id_iva) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->aplicarNuevoIva('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $unique, $id_iva, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarDescuentoProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'La Factura es obligatorio';
            if (empty($_POST['unique']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'unique', FILTER_SANITIZE_STRIPPED);
                $descuento          = filter_input(INPUT_POST, 'descuento', FILTER_SANITIZE_STRING) ?? 0;
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->aplicarDescuentoProducto('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $unique, $descuento, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function aplicarDescuentoAdicional()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura'])) {
                $this->array_errors[] = 'La Factura es obligatorio';
            } else {

                $id_factura = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $descuento_global = filter_input(INPUT_POST, 'descuento', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->aplicarDescuentoAdicional('cmppeeditar', $id_factura, $data_factura['cmppe_cliid'], $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function eliminarProducto()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['id_factura']))
                $this->array_errors[] = 'El Cliente es obligatorio';
            if (empty($_POST['id_producto']))
                $this->array_errors[] = 'El Producto es obligatorio';

            if (empty($this->array_errors)) {

                $id_factura         = filter_input(INPUT_POST, 'id_factura', FILTER_VALIDATE_INT);
                $unique             = filter_input(INPUT_POST, 'id_producto', FILTER_SANITIZE_STRING);
                $descuento_global   = filter_input(INPUT_POST, 'descuento_global', FILTER_SANITIZE_STRING) ?? 0;

                if ($id_factura) {
                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);
                    if ($data_factura) {
                        if ($this->esEditable($id_factura)) {
                            $result = $this->carritoCompras->eliminarProducto('cmppeeditar', 'app_cmppedidos', $id_factura, $data_factura['cmppe_cliid'], $unique, $descuento_global);
                            if (isset($result['errors'])) {
                                return Http::json_response(['errors' => $result['errors']]);
                            } else {
                                return Http::json_response($result);
                            }
                        } else {
                            $this->array_errors[] = 'Acción prohibida';
                        }
                    } else {
                        $this->array_errors[] = 'Acción prohibida';
                    }
                } else {
                    $this->array_errors[] = 'Acción prohibida';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function guardarDetallesCliente()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['prov_cifnif']))
                $this->array_errors[] = 'El campo Cedula de identidad es obligatorio';
            if (empty($_POST['prov_tipoidfiscal']))
                $this->array_errors[] = 'El campo Tipo de documento es obligatorio';
            if (empty($_POST['prov_razonsocial']))
                $this->array_errors[] = 'El campo Razón social es obligatorio';
            if (empty($_POST['prov_direccion']))
                $this->array_errors[] = 'El campo Dirección es obligatorio';

            if (empty($this->array_errors)) {

                $prov_id                = filter_input(INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT);
                $prov_tipoidfiscal      = filter_input(INPUT_POST, 'prov_tipoidfiscal', FILTER_SANITIZE_NUMBER_INT);
                $prov_cifnif            = $this->solicitud->sanitize($_POST['prov_cifnif'], FILTER_SANITIZE_STRING);
                $prov_razonsocial       = $this->solicitud->sanitize($_POST['prov_razonsocial'], FILTER_SANITIZE_STRING);
                $prov_direccion         = $this->solicitud->sanitize($_POST['prov_direccion'], FILTER_SANITIZE_STRING);
                $prov_paisid            = filter_input(INPUT_POST, 'prov_paisid', FILTER_SANITIZE_NUMBER_INT);
                $prov_estado            = $this->solicitud->sanitize($_POST['prov_estado'], FILTER_SANITIZE_STRING);
                $prov_ciudad            = $this->solicitud->sanitize($_POST['prov_ciudad'], FILTER_SANITIZE_STRING);
                $prov_codigo_postal     = $this->solicitud->sanitize($_POST['prov_codigo_postal'], FILTER_SANITIZE_STRING);

                $result = $this->carritoCompras->editarDetallesCliente($prov_id, [
                    'prov_tipoidfiscal'     => $prov_tipoidfiscal,
                    'prov_cifnif'           => $prov_cifnif,
                    'prov_razonsocial'      => $prov_razonsocial,
                    'prov_direccion'        => $prov_direccion,
                    'prov_paisid'           => $prov_paisid,
                    'prov_estado'           => $prov_estado,
                    'prov_ciudad'           => $prov_ciudad,
                    'prov_codigo_postal'    => $prov_codigo_postal
                ]);

                if (isset($result['errors'])) {
                    return Http::json_response(['errors' => $result['errors']]);
                } else {
                    return Http::json_response($result);
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function guardarEnvioCliente()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_factura             = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $env_nombre_opcional    = filter_input(INPUT_POST, 'env_nombre_opcional', FILTER_SANITIZE_STRING);
            $env_agencia_transporte = filter_input(INPUT_POST, 'env_agencia_transporte', FILTER_SANITIZE_STRING);
            $env_codigo_seguimiento = filter_input(INPUT_POST, 'env_codigo_seguimiento', FILTER_SANITIZE_STRING);
            $env_direccion          = filter_input(INPUT_POST, 'env_direccion', FILTER_SANITIZE_STRING);
            $env_pais               = filter_input(INPUT_POST, 'env_pais', FILTER_SANITIZE_NUMBER_INT);
            $env_estado             = filter_input(INPUT_POST, 'env_estado', FILTER_SANITIZE_STRING);
            $env_ciudad             = filter_input(INPUT_POST, 'env_ciudad', FILTER_SANITIZE_STRING);
            $env_codigopostal       = filter_input(INPUT_POST, 'env_codigopostal', FILTER_SANITIZE_STRING);
            $env_asegurado          = ('true' == filter_input(INPUT_POST, 'env_asegurado')) ? 1 : 0;

            if (!empty($id_factura)) {

                if (empty($env_agencia_transporte))
                    $this->array_errors[] = "El campo Agencia de tranporte es obligatorio.";
                if (empty($env_codigo_seguimiento))
                    $this->array_errors[] = "El campo Codigo de seguimiento es obligatorio.";
                if (empty($env_direccion))
                    $this->array_errors[] = "El campo Dirección obligatorio.";
                if (empty($env_pais))
                    $this->array_errors[] = "El campo País es obligatorio.";
                if (empty($env_estado))
                    $this->array_errors[] = "El campo Estado es obligatorio.";
                if (empty($env_ciudad))
                    $this->array_errors[] = "El campo Ciudad es obligatorio.";
                if (empty($env_codigopostal))
                    $this->array_errors[] = "El campo Codigo postal es obligatorio.";

                if (empty($this->array_errors)) {

                    $data_factura = $this->notas_entrega_model->obtenerFactura($id_factura);

                    if ($data_factura['cmppe_estatus'] != 1) {
                        Http::json_response($this->notas_entrega_model->editar($id_factura, [
                            'cmppe_env_nombres'            => $env_nombre_opcional ?? $data_factura['cmppe_env_nombres'],
                            'cmppe_env_direccion'          => $env_direccion,
                            'cmppe_env_pais'               => $env_pais,
                            'cmppe_env_estado'             => $env_estado,
                            'cmppe_env_ciudad'             => $env_ciudad,
                            'cmppe_env_codigo_postal'      => $env_codigopostal,
                            'cmppe_env_agencia_transporte' => $env_agencia_transporte,
                            'cmppe_env_codigo_seguimiento' => $env_codigo_seguimiento,
                            'cmppe_env_asegurado'          => $env_asegurado,
                        ]));
                    } else {
                        $this->array_errors[] = "La factura no se puede editar ya que ha sigo pagada en su totalidad.";
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function comprobarFactura($id_factura)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $id_factura = filter_var($id_factura, FILTER_SANITIZE_NUMBER_INT);
            $factura = $this->notas_entrega_model->obtenerFactura($id_factura);
            if ($factura['cmppe_estatus'] == 1) {
                $this->array_errors[] = 'Ya la factura a sido pagada en su totalidad';
            } else {
                Http::json_response(true);
            }
            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    private function esEditable($id_factura)
    {
        $factura = $this->notas_entrega_model->obtenerFactura($id_factura);
        return ($factura['cmppe_estatus'] == 1) ? false : true;
    }

    public function ventaEnEsperaEditar()
    {
        if ($_SERVER['REQUEST_METHOD'] == 'POST' && isAjax()) {

            $id_serie       = filter_input(INPUT_POST, 'id_serie', FILTER_SANITIZE_NUMBER_INT);
            $id_factura     = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $fecha          = filter_input(INPUT_POST, 'fecha', FILTER_SANITIZE_STRING);
            $observaciones  = filter_input(INPUT_POST, 'observaciones', FILTER_SANITIZE_STRING) ?? '';
            $afectar_stock  = false;
            $afectar_precio = false;
            $cantidad_agregada = 0;

            if (!empty($id_factura)) {

                if (empty($id_serie))
                    $this->array_errors[] = 'El campo Serie es obligatorio.';
                if (empty($fecha))
                    $this->array_errors[] = 'El campo fecha es obligatorio.';

                if (empty($this->array_errors)) {

                    $serie      = $this->notas_entrega_model->obtenerSerie($id_serie);
                    $factura    = $this->notas_entrega_model->obtenerFactura($id_factura);
                    $proveedor    = $this->notas_entrega_model->obtenerProveedor($factura['cmppe_cliid']);

                    if ($proveedor && $serie && $factura && $this->esEditable($id_factura)) {
                        $divisa_master = Moneda::Predeterminada();
                        $new_productos = new Productos();
                        $notas_entrega_articulos_model = new CmppedidosArticulos();
                        $contenido_carrito = $this->carritoCompras->contenidoCarrito('cmppeeditar', $id_factura);
                        $fecha = date('Y-m-d', strtotime($fecha)) . ' ' . date('H:i:s');

                        if (empty($this->array_errors)) {

                            if ($this->notas_entrega_model->editar($id_factura, [
                                'cmppe_subtotal'          => $_SESSION["cmppeeditar$id_factura"]['subtotal'],
                                'cmppe_descuento'         => $_SESSION["cmppeeditar$id_factura"]['descuento'],
                                'cmppe_neto'              => $_SESSION["cmppeeditar$id_factura"]['neto'],
                                'cmppe_iva'               => $_SESSION["cmppeeditar$id_factura"]['iva'],
                                'cmppe_total'             => $_SESSION["cmppeeditar$id_factura"]['total'],
                                'cmppe_articulos_total'   => $_SESSION["cmppeeditar$id_factura"]['articulos_cantidad'],
                                'cmppe_divid'             => $_SESSION["cmppeeditar$id_factura"]['divisa'],
                                'cmppe_serid'             => $serie['ser_id'],
                                'cmppe_serie'             => $serie['ser_descripcion'],
                                'cmppe_empresaid'         => 1,
                                'cmppe_empleadoid'        => $_SESSION['user_data']['einfo_id'],
                                'cmppe_fecha'             => $fecha,
                                'cmppe_cliid'             => $proveedor['prov_id'],
                                'cmppe_env_nombres'       => $proveedor['prov_razonsocial'],
                                'cmppe_env_direccion'     => $proveedor['prov_direccion'],
                                'cmppe_env_pais'          => $proveedor['prov_paisid'],
                                'cmppe_env_estado'        => $proveedor['prov_estado'],
                                'cmppe_env_ciudad'        => $proveedor['prov_ciudad'],
                                'cmppe_env_codigo_postal' => $proveedor['prov_codigo_postal'],
                                'cmppe_estatus'           => 2,
                                'cmppe_observaciones'     => $observaciones
                            ])) {

                                foreach ($contenido_carrito as $row) {
                                    $detalles_producto = $notas_entrega_articulos_model->obtenerDetalleProductoFacturado($id_factura, $row['id'], $row['id_almacen']);
                                    $almacen  = $this->notas_entrega_model->obtenerAlmacen($row['id_almacen']);

                                    if ($detalles_producto) {
                                        if ($notas_entrega_articulos_model->editar($detalles_producto['cmppeart_id'], [
                                            'cmppeart_procodigo'              => $row['codigo'],
                                            'cmppeart_propnombre'             => $row['nombre'],
                                            'cmppeart_proprecio'              => $row['costo'],
                                            'cmppeart_procantidad'            => $row['cantidad'],
                                            'cmppeart_prodescuento'           => $row['descuento'],
                                            'cmppeart_proneto'                => $row['neto'],
                                            'cmppeart_idiva'                  => $row['id_iva'],
                                            'cmppeart_almid'                  => $row['id_almacen'],
                                            'cmppeart_proiva'                 => $row['iva'],
                                            'cmppeart_ivatotal'               => $row['iva_total'],
                                            'cmppeart_prototal'               => $row['total']
                                        ])) {
                                            $precio = TarifasControlador::aplicar(null, $row['costo'], "pvp", $row['descuento'], 0,0);
                                            //DESCUENTO PRODUCTO
                                            $new_productos->editar($row['id'], [
                                                'pro_costo' => $precio,
                                                //'pro_impid' => $row['id_iva']
                                            ]);
                                        }
                                    } else {
                                        if ($notas_entrega_articulos_model->guardar([
                                            'cmppeart_peid'                  => $id_factura,
                                            'cmppeart_proid'                  => $row['id'],
                                            'cmppeart_procodigo'              => $row['codigo'],
                                            'cmppeart_propnombre'             => $row['nombre'],
                                            'cmppeart_proprecio'              => $row['costo'],
                                            'cmppeart_procantidad'            => $row['cantidad'],
                                            'cmppeart_prodescuento'           => $row['descuento'],
                                            'cmppeart_proneto'                => $row['neto'],
                                            'cmppeart_idiva'                  => $row['id_iva'],
                                            'cmppeart_almid'                  => $row['id_almacen'],
                                            'cmppeart_proiva'                 => $row['iva'],
                                            'cmppeart_ivatotal'               => $row['iva_total'],
                                            'cmppeart_prototal'               => $row['total']
                                        ], true)) {
                                            $precio = TarifasControlador::aplicar(null, $row['costo'], "pvp", $row['descuento'], 0,0);
                                            //DESCUENTO PRODUCTO
                                            if ($_SESSION["cmppeeditar$id_factura"]['divisa'] != $divisa_master['id']) {
                                                $factor =  $this->carritoCompras->obtenerPrecioSegunDivisa($precio, $_SESSION["cmppeeditar$id_factura"]['divisa'], $divisa_master['id']);
                                                $new_productos->editar($row['id'], [
                                                    'pro_costo' => $factor['precio'],
                                                    // 'pro_impid' => $row['id_iva']
                                                ]);
                                            } else {
                                                $new_productos->editar($row['id'], [
                                                    'pro_costo' => $precio,
                                                    //'pro_impid' => $row['id_iva']
                                                ]);
                                            }
                                        }
                                    }
                                }

                                Http::json_response(ruta_base() . 'cmppedidoseditar/verFactura/' . $id_factura);
                            } else {
                                $this->array_errors[] = "Los siento ocurrio un error al guardar la venta";
                            }
                        }
                    }
                }

                Http::json_response(['errors' => $this->array_errors]);
            }
        }
    }

    public function obtenerMetodosSegunDivisa()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_divisa  = filter_input(INPUT_POST, 'id_divisa', FILTER_SANITIZE_NUMBER_INT);
            $id_factura = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $factor = 1;

            if (empty($id_divisa))
                $this->array_errors[] = 'Debe seleccionar una divisa';

            if (!empty($id_factura) && empty($this->array_errors)) {

                $factura        = $this->notas_entrega_model->obtenerFactura($id_factura);
                $divisa         = Moneda::obtenerDivisa($id_divisa);

                if ($divisa && $factura) {

                    $data_metodos   = $this->notas_entrega_model->obtenerMetodosSegunDivisa($id_divisa);

                    if ($data_metodos) {

                        if ($factura['cmppe_divid'] != $divisa['id']) {
                            $divisa_factor = $this->notas_entrega_model->obtenerDivisaAlCambio($factura['cmppe_divid'], $divisa['id']);
                            if ($divisa_factor) {
                                $factor = $divisa_factor['df_factor'];
                            } else {
                                $this->array_errors[] = 'Lo siento no se encuentra ningun valor de conversión para la moneda que intenta escoger y la moneda en que se realizo la factura';
                            }
                        }

                        if (empty($this->array_errors)) {

                            $movimiento_model = new MovimientoModel();
                            $total_pagos = $movimiento_model->totalPagos($factura['cmppe_id'], 10, 2);
                            $total = round(($factura['cmppe_total'] + ($total_pagos['total'] ?? 0)), 2);

                            Http::json_response([
                                'factor'            => $factor,
                                'divisa'            => $divisa,
                                'data_metodos'      => $data_metodos,
                                'restante'          => $total
                            ]);
                        }
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerCuentasSegunMetodo($id_metodo_divisa)
    {
        if ('GET' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_metodo_divisa = filter_var($id_metodo_divisa, FILTER_SANITIZE_NUMBER_INT);

            if (empty($id_metodo_divisa))
                $this->array_errors[] = 'Debe seleccionar un metodo';

            if (empty($this->array_errors)) {
                $data = $this->notas_entrega_model->obtenerCuentasSegunMetodo($id_metodo_divisa);
                if ($data) {
                    Http::json_response($data);
                } else {
                    $this->array_errors[] = 'Lo siento no hay cuentas registradas para este metoto de pago';
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerSaldoSegunCuenta()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $id = filter_input(INPUT_GET, 'id_cuenta', FILTER_SANITIZE_NUMBER_INT);
            $divisa_cuenta = $this->notas_entrega_model->obtenerDivisaSegunCuenta($id);
            $cuenta = $this->notas_entrega_model->consultar("SELECT cmp_cueid FROM app_cuentas_metodopago WHERE cmp_id = $id")->row();
            $saldo = $this->notas_entrega_model->consultar("SELECT bc_saldo FROM app_bancos_cuentas WHERE bc_id = {$cuenta['cmp_cueid']}")->row();
            $saldo = $saldo['bc_saldo'] ?? 0;
            Http::json_response([
                'div_locale'    => $divisa_cuenta['div_locale'],
                'div_simbolo'   => $divisa_cuenta['div_simbolo'],
                'saldo'         => $saldo
            ]);
        }
    }

    public function eniviarFactura()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id     = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $de     = filter_input(INPUT_POST, 'de', FILTER_SANITIZE_EMAIL);
            $para   = filter_input(INPUT_POST, 'para', FILTER_SANITIZE_EMAIL);
            $copia  = filter_input(INPUT_POST, 'copia', FILTER_SANITIZE_EMAIL);
            $firma  = filter_input(INPUT_POST, 'firma', FILTER_SANITIZE_EMAIL);
            $tipo   = filter_input(INPUT_POST, 'tipo', FILTER_SANITIZE_NUMBER_INT) ?? 1;

            if (empty($de))
                $this->array_errors[] = "El campo De es obligatorio.";
            if (empty($para))
                $this->array_errors[] = "El campo Para es obligatorio.";

            if (empty($this->array_errors) && !empty($id)) {

                $empresa_data       = $this->notas_entrega_model->obtenerDatosEmpresa();

                switch ($tipo) {
                    case 1:
                        $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);

                        $articulos_model    = new CmppedidosArticulos();
                        $articulos_data     = $articulos_model->articulos($factura_data['cmppe_id']);
                        $proveedor_data       = $this->notas_entrega_model->obtenerProveedor($factura_data['cmppe_cliid']);
                       
                        $documento_data     = $this->notas_entrega_model->obtenerDocumento(10);

                        $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['cmppe_correlativo']);

                        $mpdf = new \Mpdf\Mpdf([
                            'mode'              => 'utf-8',
                            'margin_left'       => 15,
                            'margin_right'      => 15,
                            'margin_top'        => 35,
                            'margin_bottom'     => 30,
                            'margin_header'     => 15,
                            'margin_footer'     => 30,
                            'format'            => [210, 148.5],
                            'orientation'       => 'P',
                            'default_font_size' => 8,
                            'default_font'      => 'helvetica'
                        ]);

                        $mpdf->SetTitle($mascara);
                        $mpdf->SetAuthor($empresa_data['emp_nombre']);
                        $mpdf->SetCreator($empresa_data['emp_nombre']);
                        $mpdf->SetSubject($mascara);

                        $html_header = Vista::viewPDF('pdfcompras/pedidos/header', [
                            'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
                            'emp_rif'       => $empresa_data['emp_rif'] ?? '',
                            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                            'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
                            'emp_estado'    => $empresa_data['emp_estado'] ?? '',
                            'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
                        ]);

                        $mpdf->SetHTMLHeader($html_header);

                        $html_body = Vista::viewPDF('pdfcompras/pedidos/body_completo', [
                            'factura'       => $mascara,
                            'cliente'       => $proveedor_data['prov_razonsocial'],
                            'direccion'     => $proveedor_data['prov_direccion'],
                            'fecha'         => date('Y-m-d', strtotime($factura_data['cmppe_fecha'])),
                            'cedularif'     => $proveedor_data['prov_cifnif'],
                            'telefono'      => $proveedor_data['prov_telefono1'],
                            'div_locale'    => $factura_data['div_locale'],
                            'div_simbolo'  => $factura_data['div_simbolo'],
                            'articulos'     => $articulos_data,
                            'observaciones' => $factura_data['cmppe_observaciones']
                        ]);

                        $mpdf->WriteHTML($html_body);

                        $html_footer = Vista::viewPDF('pdfcompras/pedidos/footer', [
                            'div_locale'    => $factura_data['div_locale'],
                            'div_simbolo'  => $factura_data['div_simbolo'],
                            'total'         => $factura_data['cmppe_total'],
                            'iva'           => $factura_data['cmppe_iva'],
                            'neto'          => $factura_data['cmppe_neto'],
                            'descuento'     => $factura_data['cmppe_descuento']
                        ]);

                        break;
                    case 2:
                        $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);

                        $articulos_model    = new CmppedidosArticulos();
                        $articulos_data     = $articulos_model->articulos($factura_data['cmppe_id']);
                        $proveedor_data       = $this->notas_entrega_model->obtenerProveedor($factura_data['cmppe_cliid']);
                        
                        $documento_data     = $this->notas_entrega_model->obtenerDocumento(10);

                        $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['cmppe_correlativo']);

                        $mpdf = new \Mpdf\Mpdf([
                            'mode'              => 'utf-8',
                            'margin_left'       => 15,
                            'margin_right'      => 15,
                            'margin_top'        => 35,
                            'margin_bottom'     => 30,
                            'margin_header'     => 15,
                            'margin_footer'     => 30,
                            'format'            => [210, 148.5],
                            'orientation'       => 'P',
                            'default_font_size' => 8,
                            'default_font'      => 'helvetica'
                        ]);

                        $mpdf->SetTitle($mascara);
                        $mpdf->SetAuthor($empresa_data['emp_nombre']);
                        $mpdf->SetCreator($empresa_data['emp_nombre']);
                        $mpdf->SetSubject($mascara);

                        $html_header = Vista::viewPDF('pdfcompras/pedidos/header', [
                            'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
                            'emp_rif'       => $empresa_data['emp_rif'] ?? '',
                            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
                            'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
                            'emp_estado'    => $empresa_data['emp_estado'] ?? '',
                            'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
                        ]);

                        $mpdf->SetHTMLHeader($html_header);

                        $html_body = Vista::viewPDF('pdfcompras/pedidos/body', [
                            'factura'       => $mascara,
                            'cliente'       => $proveedor_data['prov_razonsocial'],
                            'direccion'     => $proveedor_data['prov_direccion'],
                            'fecha'         => date('Y-m-d', strtotime($factura_data['cmppe_fecha'])),
                            'cedularif'     => $proveedor_data['prov_cifnif'],
                            'telefono'      => $proveedor_data['prov_telefono1'],
                            'div_locale'    => $factura_data['div_locale'],
                            'div_simbolo'  => $factura_data['div_simbolo'],
                            'articulos'     => $articulos_data,
                            'observaciones' => $factura_data['cmppe_observaciones']
                        ]);

                        $mpdf->WriteHTML($html_body);

                        $html_footer = Vista::viewPDF('pdfcompras/pedidos/footer_p', [
                            'div_locale'    => $factura_data['div_locale'],
                            'div_simbolo'  => $factura_data['div_simbolo'],
                            'total'         => $factura_data['cmppe_total'],
                            'iva'           => $factura_data['cmppe_iva'],
                            'neto'          => $factura_data['cmppe_neto'],
                            'descuento'     => $factura_data['cmppe_descuento']
                        ]);

                        break;
                }
               
                $mpdf->SetHTMLFooter($html_footer);
                $content = $mpdf->Output('', 'S');
               
                $mail = new PHPMailer(true);
                try {

                    $mail->isSMTP();
                    $mail->Host       = "{$empresa_data['emp_host_correo']}";
                    $mail->SMTPAuth   = true;
                    $mail->Username   = "{$empresa_data['emp_email']}";
                    $mail->Password   = "{$empresa_data['emp_email_password']}";
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
                    $mail->Port       = $empresa_data['emp_port_correo'];

                    $mail->setFrom($de);
                    $mail->addAddress($para);
                    $mail->addReplyTo($de, 'Empresa');

                    if (!empty($copia)) {
                        $mail->addAddress($copia);
                    }

                    $mail->addStringAttachment($content, "$mascara.pdf");

                    $mail->isHTML(true);
                    $mail->Subject = 'PEDIDO';
                    $mail->Body    = "$mascara";

                    $mail->send();

                    Http::json_response("Envio de factura exitosa");
                } catch (Exception $e) {
                    $this->array_errors[] = "Message could not be sent. Mailer Error: {$mail->ErrorInfo}.";
                }
            }

            Http::json_response([
                'errors' => $this->array_errors
            ]);
        }
    }

    private function pdfCompleto($id_factura)
    {
        $factura_data       = $this->notas_entrega_model->obtenerFacturaCompleta($id_factura);

        $articulos_model    = new CmppedidosArticulos();
        $articulos_data     = $articulos_model->articulos($factura_data['cmppe_id']);
        $proveedor_data       = $this->notas_entrega_model->obtenerProveedor($factura_data['cmppe_cliid']);
        $empresa_data       = $this->notas_entrega_model->obtenerDatosEmpresa();
        $documento_data     = $this->notas_entrega_model->obtenerDocumento(10);

        $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['cmppe_correlativo']);

        $mpdf = new \Mpdf\Mpdf([
            'mode'              => 'utf-8',
            'margin_left'       => 15,
            'margin_right'      => 15,
            'margin_top'        => 35,
            'margin_bottom'     => 30,
            'margin_header'     => 15,
            'margin_footer'     => 30,
            'format'            => 'A4',
            'orientation'       => 'P',
            'default_font_size' => 8,
            'default_font'      => 'helvetica'
        ]);

        $mpdf->SetTitle($mascara);
        $mpdf->SetAuthor($empresa_data['emp_nombre']);
        $mpdf->SetCreator($empresa_data['emp_nombre']);
        $mpdf->SetSubject($mascara);

        $html_header = Vista::viewPDF('pdfcompras/pedidos/header.php', [
            'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
            'emp_rif'       => $empresa_data['emp_rif'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
            'emp_estado'    => $empresa_data['emp_estado'] ?? '',
            'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
        ]);

        $mpdf->SetHTMLHeader($html_header);

        $html_body = Vista::viewPDF('pdfcompras/pedidos/body_completo', [
            'factura'       => $mascara,
            'cliente'       => $proveedor_data['prov_razonsocial'],
            'direccion'     => $proveedor_data['prov_direccion'],
            'fecha'         => date('Y-m-d', strtotime($factura_data['cmppe_fecha'])),
            'cedularif'     => $proveedor_data['prov_cifnif'],
            'telefono'      => $proveedor_data['prov_telefono1'],
            'div_locale'    => $factura_data['div_locale'],
            'div_simbolo'  => $factura_data['div_simbolo'],
            'articulos'     => $articulos_data,
            'observaciones' => $factura_data['cmppe_observaciones']
        ]);

        $mpdf->WriteHTML($html_body);

        $html_footer = Vista::viewPDF('pdfcompras/pedidos/footer', [
            'div_locale'    => $factura_data['div_locale'],
            'div_simbolo'  => $factura_data['div_simbolo'],
            'total'         => $factura_data['cmppe_total'],
            'iva'           => $factura_data['cmppe_iva'],
            'neto'          => $factura_data['cmppe_neto'],
            'descuento'     => $factura_data['cmppe_descuento']
        ]);

        $mpdf->SetHTMLFooter($html_footer);

        return [
            'mpdf'      => $mpdf,
            'mascara'   => $mascara . '.pdf',
            'empresa'   => $empresa_data,
        ];
    }

    private function pdfMediaPagina($id_factura)
    {
        $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id_factura);

        $articulos_model    = new CmppedidosArticulos();
        $articulos_data     = $articulos_model->articulos($factura_data['cmppe_id']);
        $proveedor_data       = $this->notas_entrega_model->obtenerProveedor($factura_data['cmppe_cliid']);
        $empresa_data       = $this->notas_entrega_model->obtenerDatosEmpresa();
        $documento_data     = $this->notas_entrega_model->obtenerDocumento(10);

        $mascara = mascara_correlativo($documento_data['doc_prefijo'], $factura_data['ser_impuesto'], $factura_data['cmppe_correlativo']);

        $mpdf = new \Mpdf\Mpdf([
            'mode'              => 'utf-8',
            'margin_left'       => 15,
            'margin_right'      => 15,
            'margin_top'        => 35,
            'margin_bottom'     => 30,
            'margin_header'     => 15,
            'margin_footer'     => 30,
            'format'            => [210, 148.5],
            'orientation'       => 'P',
            'default_font_size' => 8,
            'default_font'      => 'helvetica'
        ]);

        $mpdf->SetTitle($mascara);
        $mpdf->SetAuthor($empresa_data['emp_nombre']);
        $mpdf->SetCreator($empresa_data['emp_nombre']);
        $mpdf->SetSubject($mascara);

        $html_header = Vista::viewPDF('pdfcompras/pedidos/header', [
            'emp_nombre'    => $empresa_data['emp_nombre'] ?? '',
            'emp_rif'       => $empresa_data['emp_rif'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_direccion' => $empresa_data['emp_direccion'] ?? '',
            'emp_codpostal' => $empresa_data['emp_codpostal'] ?? '',
            'emp_estado'    => $empresa_data['emp_estado'] ?? '',
            'emp_ciudad'    => $empresa_data['emp_ciudad'] ?? '',
        ]);

        $mpdf->SetHTMLHeader($html_header);

        $html_body = Vista::viewPDF('pdfcompras/pedidos/body', [
            'factura'       => $mascara,
            'cliente'       => $proveedor_data['prov_razonsocial'],
            'direccion'     => $proveedor_data['prov_direccion'],
            'fecha'         => date('Y-m-d', strtotime($factura_data['cmppe_fecha'])),
            'cedularif'     => $proveedor_data['prov_cifnif'],
            'telefono'      => $proveedor_data['prov_telefono1'],
            'div_locale'    => $factura_data['div_locale'],
            'div_simbolo'  => $factura_data['div_simbolo'],
            'articulos'     => $articulos_data,
            'observaciones' => $factura_data['cmppe_observaciones']
        ]);

        $mpdf->WriteHTML($html_body);

        $html_footer = Vista::viewPDF('pdfcompras/pedidos/footer_p', [
            'div_locale'    => $factura_data['div_locale'],
            'div_simbolo'  => $factura_data['div_simbolo'],
            'total'         => $factura_data['cmppe_total'],
            'iva'           => $factura_data['cmppe_iva'],
            'neto'          => $factura_data['cmppe_neto'],
            'descuento'     => $factura_data['cmppe_descuento']
        ]);

        $mpdf->SetHTMLFooter($html_footer);

        return [
            'mpdf'      => $mpdf,
            'mascara'   => $mascara . '.pdf',
            'empresa'   => $empresa_data,
        ];
    }

    public function pdf($id)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD']) {

            $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);

            if (!$factura_data) {
                Http::notFound();
            }

            $mpdf = $this->pdfCompleto($id);

            $mpdf['mpdf']->Output("{$mpdf['mascara']}", \Mpdf\Output\Destination::INLINE);
            exit;
        }
    }

    public function pdfMP($id)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD']) {

            $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $factura_data = $this->notas_entrega_model->obtenerFacturaCompleta($id);

            if (!$factura_data) {
                Http::notFound();
            }

            $mpdf = $this->pdfMediaPagina($id);

            $mpdf['mpdf']->Output("{$mpdf['mascara']}", \Mpdf\Output\Destination::INLINE);
            exit;
        }
    }

    public function enviarDocumento()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $new_cmpfacdoc = new CmpPedidosDocumentos();

            $id_factura     = filter_input(INPUT_POST, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
            $descripcion    = filter_input(INPUT_POST, 'descripcion', FILTER_SANITIZE_STRING);

            if (!empty($id_factura)) {

                if (empty($descripcion))
                    $this->array_errors[] = "El campo Descripcion es obligatorio.";
                if (empty($_FILES['documento']))
                    $this->array_errors[] = "El documento es obligatorio.";

                if (empty($this->array_errors)) {

                    $fileTmpPath    = $_FILES['documento']['tmp_name'];
                    $fileName       = $_FILES['documento']['name'];
                    $fileSize       = $_FILES['documento']['size'];
                    $extenciones    = array('jpg', 'jpeg', 'png', 'pdf');

                    if ($_FILES['documento']['error'] === UPLOAD_ERR_OK) {

                        $fileNameCmps = explode(".", basename($_FILES['documento']['name']));
                        $fileExtension = strtolower(end($fileNameCmps));
                        $newFileName = md5(time() . $fileName) . '.' . $fileExtension;
                        $dest_path = 'upload/comprasnotas/' . $newFileName;

                        if ($fileSize < 1000000) {
                            if (in_array($fileExtension, $extenciones)) {
                                if (move_uploaded_file($fileTmpPath, $dest_path)) {
                                    if ($new_cmpfacdoc->guardar([
                                        'cmppedoc_cmppeid'          => $id_factura,
                                        'cmppedoc_descripcion'    => $descripcion,
                                        'cmppedoc_documento'      => $newFileName
                                    ])) {
                                        Http::json_response('Archivo subido satisfactoriamente');
                                    } else {
                                        unlink($dest_path);
                                        $this->array_errors[] = "Se produjo un error al cargar el archivo";
                                    }
                                } else {
                                    $this->array_errors[] = "Se produjo un error al cargar el archivo";
                                }
                            } else {
                                $this->array_errors[] = "Solo archivos con formato PDF, PNG o JPG";
                            }
                        } else {
                            $this->array_errors[] = "EL archivo supera el limite de 1MB";
                        }
                    } else {
                        $this->array_errors[] = "Se produjo un error al cargar el archivo";
                    }
                }
            }

            Http::json_response([
                'errors' => $this->array_errors
            ]);
        }
    }

    public function obtnerDocumentos($id_factura)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $new_cmpfacdoc = new CmppedidosDocumentos();

            $id_factura = filter_var($id_factura, FILTER_SANITIZE_NUMBER_INT);

            if (!empty($id_factura)) {
                Http::json_response($new_cmpfacdoc->obtenerDocumentos($id_factura));
            }
        }
    }

    public function cambiarProveedor()
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET' && isAjax()) {

            $id_proveedor   = filter_input(INPUT_GET, 'id_proveedor', FILTER_SANITIZE_NUMBER_INT);
            $id_factura     = filter_input(INPUT_GET, 'id_factura', FILTER_SANITIZE_NUMBER_INT);

            if (!empty($id_proveedor) && !empty($id_factura)) {

                $proveedor = $this->notas_entrega_model->obtenerProveedor($id_proveedor);
                $factura = $this->notas_entrega_model->obtenerFactura($id_factura);

                if ($proveedor && $factura) {
                    if ($this->esEditable($id_factura)) {
                        if ($this->notas_entrega_model->editar($id_factura, [
                            'cmppe_cliid'             => $proveedor['prov_id'],
                            'cmppe_env_nombres'       => $proveedor['prov_razonsocial'],
                            'cmppe_env_direccion'     => $proveedor['prov_direccion'],
                            'cmppe_env_pais'          => $proveedor['prov_paisid'],
                            'cmppe_env_estado'        => $proveedor['prov_estado'],
                            'cmppe_env_ciudad'        => $proveedor['prov_ciudad'],
                            'cmppe_env_codigo_postal' => $proveedor['prov_codigo_postal'],
                        ])) {
                            Http::json_response(ruta_base() . 'cmppedidoseditar/verFactura/' . $id_factura);
                        } else {
                            $this->array_errors[] = "Los siento ocurrio un error al editar el proveedor de la nota de entrega";
                        }
                    } else {
                        $this->array_errors[] = "Nota de entrega no editable";
                    }
                }

                Http::json_response(['errors' => $this->array_errors]);
            }
        }
    }

    public function anularFactura()
    {
        if ($_SERVER['REQUEST_METHOD'] == 'GET' && isAjax()) {

            if ($this->module->has_module_action_permission('facturacion', 'anular_doc', $_SESSION['user_data']['emp_id'])) {

                $id_factura     = filter_input(INPUT_GET, 'id_factura', FILTER_SANITIZE_NUMBER_INT);
                $descripcion    = filter_input(INPUT_GET, 'descripcion', FILTER_SANITIZE_STRING);

                $factura = $this->notas_entrega_model->obtenerFactura($id_factura);

                if (!$factura) {
                    $this->array_errors[] = 'ACCESO DENEGADO';
                    return Http::json_response(['errors' => $this->array_errors]);
                }

                if ($factura['cmppe_estatus'] == 3) {
                    $this->array_errors[] = 'El pedido ya se encuentra anulada.';
                }

                if (empty($this->array_errors)) {

                    $correlativo = $this->notas_entrega_model->chuequearCorrelativo($factura['cmppe_serid'], 11);
                    $serie = $this->notas_entrega_model->obtenerSerie($factura['cmppe_serid']);

                    if ($correlativo == false) {
                        $this->array_errors[] = "No existe correlativo entre la se serie $serie[ser_descripcion] y el documento Notas de credito COMPRAS por favor dirijase a Modulo Configuración -> correlativos";
                    }

                    if (empty($this->array_errors)) {

                        $new_articulos = new CmppedidosArticulos();
                        $new_movimiento = new MovimientoModel();

                        if (empty($this->array_errors)) {

                            if ($this->notas_entrega_model->editar($factura['cmppe_id'], [
                                'cmppe_estatus' => 3,
                                'cmppe_observaciones' => $factura['cmppe_observaciones'] . ' (ANULADO ' . $descripcion . ')'
                            ])) {

                                return Http::json_response(ruta_base() . 'cmppedidoseditar/verFactura/' . $id_factura);
                            } else {
                                Http::json_response(false);
                            }
                        }
                    }
                }
            } else {
                $this->array_errors[] = 'Acceso prohibido no posee permisos para anular el documento.';
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function actualizarFactura()
    {
        if ('POST' == $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['factura']))
                $this->array_errors[] = 'El numero de la factura es obligatorio';

            if (empty($this->array_errors)) {

                $factura       = filter_input(INPUT_POST, 'factura', FILTER_SANITIZE_STRING);
                $id       = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);

                Http::json_response($this->notas_entrega_model->editar($id, [
                    'cmppe_numfactura' => $factura,
                ]));
            } else {
                $this->array_errors[] = 'Acción prohibida';
            }
        }

        Http::json_response(['errors' => $this->array_errors]);
    }
}
