<?php

namespace App\Controladores;

use App\Modelos\Almacen_Sucursales;
use App\Modelos\Almacenes;
use App\Modelos\AlmacenUbicacion;
use App\Modelos\Pais;
use Core\Controller;
use Core\Librerias\Http;
use Core\Librerias\Module;
use Core\Vista;

final class AlmacenesControlador extends Controller
{
    private $almacen_model;
    private $array_errors = [];

    public function __construct()
    {
        parent::__construct();
        $this->module = new Module('almacenes');
        $this->almacen_model = new Almacenes();
    }

    public function index()
    {
        $new_paises = new Pais();
        $almacenSucursales = new Almacen_Sucursales();

        return Vista::view('almacenes/index', [
            'paises' => $new_paises->todos(),
            'sucursales' => $almacenSucursales->obtenerSucursales()
        ]);
    }

    public function ubicacion(int $id_almacen)
    {
        return Vista::view('almacenes/ubicacion', [
            'almacen' => $id_almacen
        ]);
    }

    public function obtenerAlmacenes()
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            Http::json_response($this->almacen_model->todos());
        }
    }

    public function obtenerSucursales(int $id_almacen)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_almacen = filter_var($id_almacen, FILTER_SANITIZE_NUMBER_INT);

            $almacenSucursales = new Almacen_Sucursales();

            Http::json_response($almacenSucursales->obtenerAlmacenesSucursales($id_almacen));
        }
    }

    public function guardarSucursal()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['sucursal']))
                $this->array_errors[] = "La sucursal es obligatorio.";

            if (empty($this->array_errors)) {

                $sucursal     = $this->solicitud->sanitize($_POST['sucursal'], FILTER_SANITIZE_NUMBER_INT);
                $id     = $this->solicitud->sanitize($_POST['id'], FILTER_SANITIZE_NUMBER_INT);

                $almacenSucursales = new Almacen_Sucursales();

                if ($almacenSucursales->existeSucursal($id, $sucursal)) {
                    $this->array_errors[] = "La sucursal ya fue agregada.";
                } else {
                    $data_form = [
                        'alsu_almid' => $id,
                        'alsu_suid' => $sucursal
                    ];
                    Http::json_response($almacenSucursales->guardar($data_form, true));
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }


    public function guardar()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            if (empty($_POST['codigo']))
                $this->array_errors[] = "El campo Código es obligatorio.";
            if (empty($_POST['descripcion']))
                $this->array_errors[] = "El campo Descripción es obligatorio.";
            if (empty($_POST['telefono']))
                $this->array_errors[] = "El campo Télefono es obligatorio.";

            if (empty($this->array_errors)) {

                $alm_codigo     = $this->solicitud->sanitize($_POST['codigo'], FILTER_SANITIZE_STRING);
                $alm_nombre     = $this->solicitud->sanitize($_POST['descripcion'], FILTER_SANITIZE_STRING);
                $alm_telefono   = $this->solicitud->sanitize($_POST['telefono'], FILTER_SANITIZE_STRING);
                $alm_ciudad     = $this->solicitud->sanitize($_POST['ciudad'], FILTER_SANITIZE_STRING);
                $alm_paid       = filter_input(INPUT_POST, 'pais', FILTER_SANITIZE_NUMBER_INT);
                $alm_garantia       = filter_input(INPUT_POST, 'garantia', FILTER_SANITIZE_NUMBER_INT);
                $alm_codpostal  = $this->solicitud->sanitize($_POST['codigo_postal'], FILTER_SANITIZE_STRING);
                $alm_provincia  = $this->solicitud->sanitize($_POST['estado'], FILTER_SANITIZE_STRING);

                if (empty($_POST['id'])) {
                    if ($this->almacen_model->where(['alm_codigo' => $alm_codigo], true))
                        $this->array_errors[] = "El codigo $alm_codigo ya se encuentra en uso";
                    if ($this->almacen_model->where(['alm_nombre', $alm_nombre]))
                        $this->array_errors[] = "El nombre $alm_nombre ya se encuentra en uso";
                } else {
                    $alm_id = filter_input(INPUT_POST, 'id', FILTER_SANITIZE_NUMBER_INT);
                    if ($this->almacen_model->where(['alm_codigo' => $alm_codigo, 'alm_id!=' => $alm_id], true))
                        $this->array_errors[] = "El codigo $alm_codigo ya se encuentra en uso";
                    if ($this->almacen_model->where(['alm_nombre' => $alm_nombre, 'alm_id!=' => $alm_id], true))
                        $this->array_errors[] = "El nombre $alm_nombre ya se encuentra en uso";
                }

                if (empty($this->array_errors)) {

                    $data_form = [
                        'alm_ciudad' => $alm_ciudad,
                        'alm_paid' => $alm_paid,
                        'alm_codigo' => $alm_codigo,
                        'alm_codpostal' => $alm_codpostal,
                        'alm_nombre' => $alm_nombre,
                        'alm_provincia' => $alm_provincia,
                        'alm_telefono' => $alm_telefono,
                        'alm_garantia' => $alm_garantia
                    ];

                    if (empty($_POST['id'])) {
                        Http::json_response($this->almacen_model->guardar($data_form, true));
                    } else {
                        Http::json_response($this->almacen_model->editar($alm_id, $data_form));
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function eliminarSucursal(int $id)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id = filter_var($id, FILTER_SANITIZE_NUMBER_INT);

            $almacenSucursales=new Almacen_Sucursales();

            if ($almacenSucursales->existe('alsu_id', $id)) {
                  Http::json_response($almacenSucursales->eliminar($id));
            } else {
                $this->array_errors[] = 'Acción prohibida';
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function eliminar(int $id_almacen)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $id_almacen = filter_var($id_almacen, FILTER_SANITIZE_NUMBER_INT);

            if ($this->almacen_model->existe('alm_id', $id_almacen)) {
                if ($this->almacen_model->chequearForanea($id_almacen)) {
                    $this->array_errors[] = "EL almacen no puede ser eliminado ya que se encuentra relacionado con algunos productos.";
                } else {
                    Http::json_response($this->almacen_model->eliminar($id_almacen));
                }
            } else {
                $this->array_errors[] = 'Acción prohibida';
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerZonas(int $id_almacen)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $newAU = new AlmacenUbicacion();
            $zonas = $newAU->obtenerZonas($id_almacen);
            Http::json_response($zonas);
        }
    }

    public function llenarCombo(int $id)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $newAU = new AlmacenUbicacion();

            $combo = $newAU->obtenerCombo($id);
            $list_pasillos  = [];
            $list_armarios  = [];
            $list_secciones = [];
            $list_cajones   = [];

            if (is_countable($combo) && count($combo)) {

                foreach ($combo as $row) {

                    if ($row['tipo'] == 'Pasillos') {
                        $list_pasillos[] = [
                            'id' => $row['id'],
                            'descripcion' => $row['descripcion'],
                        ];
                    }

                    if ($row['tipo'] == 'Armarios') {
                        $list_armarios[] = [
                            'id' => $row['id'],
                            'descripcion' => $row['descripcion'],
                        ];
                    }

                    if ($row['tipo'] == 'Secciones') {
                        $list_secciones[] = [
                            'id' => $row['id'],
                            'descripcion' => $row['descripcion'],
                        ];
                    }

                    if ($row['tipo'] == 'Cajones') {
                        $list_cajones[] = [
                            'id' => $row['id'],
                            'descripcion' => $row['descripcion'],
                        ];
                    }
                }
            }

            $list_combo = [
                'pasillos' => $list_pasillos,
                'armarios' => $list_armarios,
                'secciones' => $list_secciones,
                'cajones' => $list_cajones
            ];

            Http::json_response($list_combo);
        }
    }

    public function guardarUbicacion()
    {
        if ('POST' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $newAU = new AlmacenUbicacion();

            if (empty($_POST['descripcion']))
                $this->array_errors[] = 'El campo Descripcion es obligatorio.';
            if (empty($_POST['almacen']))
                $this->array_errors[] = 'El campo Almacen es obligatorio.';
            if (empty($_POST['tipo']))
                $this->array_errors[] = 'El campo Tipo es obligatorio.';

            if (empty($this->array_errors)) {

                $descripcion = $this->solicitud->sanitize($_POST['descripcion']);
                $almacen = $this->solicitud->sanitize($_POST['almacen'], FILTER_SANITIZE_NUMBER_INT);
                $tipo = $this->solicitud->sanitize($_POST['tipo'], FILTER_SANITIZE_STRING);
                $padre = $_POST['padre'];

                if ($tipo == 'Zonas') {
                    $padre = 0;
                }

                if ($tipo != 'Zonas' && empty($padre)) {
                    $this->array_errors[] = 'Error de procedimiento.';
                } else {
                    if ($newAU->guardar([
                        'au_descripcion' => $descripcion,
                        'au_padre'  => $padre,
                        'au_almid' => $almacen,
                        'au_tipo' => $tipo
                    ], true)) {
                        $recargar = $newAU->obtenerComboTipo($padre, $tipo);
                        Http::json_response([
                            'tipo' => $tipo,
                            'recargar' => $recargar
                        ]);
                    } else {
                        Http::json_response(false);
                    }
                }
            }

            Http::json_response(['errors' => $this->array_errors]);
        }
    }

    public function obtenerUbicaciones(int $id_alamcen)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {

            $newAU = new AlmacenUbicacion();

            $zonas = $newAU->obtenerZonasAlmacen($id_alamcen);

            $list_zonas = [];

            if (is_countable($zonas) && count($zonas)) {

                foreach ($zonas as $row) {

                    $list_pasillos  = [];
                    $list_armarios  = [];
                    $list_secciones = [];
                    $list_cajones   = [];

                    $pasillos = $newAU->obtenerComboTipo($row['id'], 'Pasillos');
                    $armarios = $newAU->obtenerComboTipo($row['id'], 'Armarios');
                    $secciones = $newAU->obtenerComboTipo($row['id'], 'Secciones');
                    $cajones = $newAU->obtenerComboTipo($row['id'], 'Cajones');

                    if (is_countable($pasillos) && count($pasillos)) {

                        foreach ($pasillos as $pasillo) {

                            $p_armarios = $newAU->obtenerComboTipo($pasillo['id'], 'Armarios');
                            $p_secciones = $newAU->obtenerComboTipo($pasillo['id'], 'Secciones');
                            $p_cajones = $newAU->obtenerComboTipo($pasillo['id'], 'Cajones');
                            $p_list_armarios = [];
                            $p_list_secciones = [];
                            $p_list_cajones = [];

                            if (is_countable($p_armarios) && count($p_armarios)) {

                                foreach ($p_armarios as $p_armario) {

                                    $p_a_secciones = $newAU->obtenerComboTipo($p_armario['id'], 'Secciones');
                                    $p_a_cajones = $newAU->obtenerComboTipo($p_armario['id'], 'Cajones');
                                    $p_a_list_secciones = [];
                                    $p_a_list_cajones = [];

                                    if (is_countable($p_a_secciones) && count($p_a_secciones)) {

                                        foreach ($p_a_secciones as $p_a_seccion) {

                                            $p_a_s_cajones = $newAU->obtenerComboTipo($p_a_seccion['id'], 'Cajones');
                                            $p_a_s_list_cajones = [];

                                            if (is_countable($p_a_s_cajones) && count($p_a_s_cajones)) {
                                                foreach ($p_a_s_cajones as $p_a_s_cajon) {
                                                    $p_a_s_list_cajones[] = [
                                                        'id' => $p_a_s_cajon['id'],
                                                        'descripcion' => $p_a_s_cajon['descripcion'],
                                                    ];
                                                }
                                            }

                                            $p_a_list_secciones[] = [
                                                'id' => $p_a_seccion['id'],
                                                'descripcion' => $p_a_seccion['descripcion'],
                                                'cajones' => $p_a_s_list_cajones
                                            ];
                                        }
                                    }

                                    if (is_countable($p_a_cajones) && count($p_a_cajones)) {
                                        foreach ($p_a_cajones as $p_a_cajon) {
                                            $p_a_list_cajones[] = [
                                                'id' => $p_a_cajon['id'],
                                                'descripcion' => $p_a_cajon['descripcion'],
                                            ];
                                        }
                                    }

                                    $p_list_armarios[] = [
                                        'id' => $p_armario['id'],
                                        'descripcion' => $p_armario['descripcion'],
                                        'secciones' => $p_a_list_secciones,
                                        'cajones' => $p_a_list_cajones
                                    ];
                                }
                            }

                            if (is_countable($p_secciones) && count($p_secciones)) {

                                foreach ($p_secciones as $p_seccion) {

                                    $p_s_cajones = $newAU->obtenerComboTipo($p_seccion['id'], 'Cajones');
                                    $p_s_list_cajones = [];

                                    if (is_countable($p_s_cajones) && count($p_s_cajones)) {
                                        foreach ($p_s_cajones as $p_s_cajon) {
                                            $p_s_list_cajones[] = [
                                                'id' => $p_s_cajon['id'],
                                                'descripcion' => $p_s_cajon['descripcion'],
                                            ];
                                        }
                                    }

                                    $p_list_secciones[] = [
                                        'id' => $p_seccion['id'],
                                        'descripcion' => $p_seccion['descripcion'],
                                        'cajones' => $p_s_list_cajones
                                    ];
                                }
                            }

                            if (is_countable($p_cajones) && count($p_cajones)) {
                                foreach ($p_cajones as $p_cajon) {
                                    $p_list_cajones[] = [
                                        'id' => $p_cajon['id'],
                                        'descripcion' => $p_cajon['descripcion'],
                                    ];
                                }
                            }

                            $list_pasillos[] = [
                                'id' => $pasillo['id'],
                                'descripcion' => $pasillo['descripcion'],
                                'armarios' => $p_list_armarios,
                                'secciones' => $p_list_secciones,
                                'cajones' => $p_list_cajones
                            ];
                        }
                    }

                    if (is_countable($armarios) && count($armarios)) {

                        foreach ($armarios as $armario) {

                            $a_secciones = $newAU->obtenerComboTipo($armario['id'], 'Secciones');
                            $a_cajones = $newAU->obtenerComboTipo($armario['id'], 'cajones');
                            $a_s_list_armarios = [];
                            $a_c_list_cajones = [];

                            if (is_countable($a_secciones) && count($a_secciones)) {

                                foreach ($a_secciones as $a_seccion) {

                                    $a_s_cajones = $newAU->obtenerComboTipo($a_seccion['id'], 'cajones');
                                    $a_s_list_cajones = [];

                                    if (is_countable($a_s_cajones) && count($a_s_cajones)) {
                                        foreach ($a_s_cajones as $a_s_cajon) {
                                            $a_s_list_cajones[] = [
                                                'id' => $a_s_cajon['id'],
                                                'descripcion' => $a_s_cajon['descripcion'],
                                            ];
                                        }
                                    }

                                    $a_s_list_armarios[] = [
                                        'id' => $a_seccion['id'],
                                        'descripcion' => $a_seccion['descripcion'],
                                        'cajones' => $a_s_list_cajones
                                    ];
                                }
                            }

                            if (is_countable($a_cajones) && count($a_cajones)) {
                                foreach ($a_cajones as $a_cajon) {
                                    $a_c_list_cajones[] = [
                                        'id' => $a_cajon['id'],
                                        'descripcion' => $a_cajon['descripcion'],
                                    ];
                                }
                            }

                            $list_armarios[] = [
                                'id' => $armario['id'],
                                'descripcion' => $armario['descripcion'],
                                'secciones' => $a_s_list_armarios,
                                'cajones' => $a_c_list_cajones
                            ];
                        }
                    }

                    if (is_countable($secciones) && count($secciones)) {

                        foreach ($secciones as $seccion) {

                            $s_cajones = $newAU->obtenerComboTipo($seccion['id'], 'Cajones');
                            $s_list_cajones = [];

                            if (is_countable($s_cajones) && count($s_cajones)) {
                                foreach ($s_cajones as $s_cajon) {
                                    $s_list_cajones[] = [
                                        'id' => $s_cajon['id'],
                                        'descripcion' => $s_cajon['descripcion'],
                                    ];
                                }
                            }

                            $list_secciones[] = [
                                'id' => $seccion['id'],
                                'descripcion' => $seccion['descripcion'],
                                'cajones' => $s_list_cajones
                            ];
                        }
                    }

                    if (is_countable($cajones) && count($cajones)) {
                        foreach ($cajones as $cajon) {
                            $list_cajones[] = [
                                'id' => $cajon['id'],
                                'descripcion' => $cajon['descripcion'],
                            ];
                        }
                    }

                    $list_zonas[] = [
                        'id' => $row['id'],
                        'descripcion' => $row['descripcion'],
                        'pasillos' => $list_pasillos,
                        'armarios' => $list_armarios,
                        'secciones' => $list_secciones,
                        'cajones' => $list_cajones
                    ];
                }
            }

            Http::json_response($list_zonas);
        }
    }

    public function eliminarUbicacion(int $id_ubicacion)
    {
        if ('GET' === $_SERVER['REQUEST_METHOD'] && isAjax()) {
            $newAU = new AlmacenUbicacion();
            Http::json_response($newAU->eliminar($id_ubicacion));
        }
    }
}
